# HDGL number visualization
from math import ceil

class HDGLNumberSystem:
    def __init__(self):
        self.phi = 1.6180339887
        self.phi_power_phi = 2.6180339887
        self.P3 = 4.2360679775
        self.P4 = 6.8541019662
        self.inv_phi = 0.6180339887
        self.inv_P3 = 0.2360679775

    def hdgl_number(self, n):
        """
        Represent integer n as a combination of HDGL primitives.
        Strategy:
        - Base unit = φ
        - Multiplicative braid = P3
        - Secondary addition = φ^φ
        """
        representation = []
        remaining = n

        # Use P3 as “multiplicative block”
        if remaining >= self.P3:
            mult = int(remaining // self.P3)
            representation.append(f"{mult}*P3")
            remaining -= mult * self.P3

        # Use φ^φ as secondary addition
        if remaining >= self.phi_power_phi:
            rep = int(remaining // self.phi_power_phi)
            representation.append(f"{rep}*phi^phi")
            remaining -= rep * self.phi_power_phi

        # Use φ as smallest unit
        if remaining >= self.phi:
            rep = int(ceil(remaining / self.phi))
            representation.append(f"{rep}*phi")
            remaining = 0

        if not representation:
            representation.append("0")

        return " + ".join(representation)

# Visualize first 20 HDGL numbers
hdgl = HDGLNumberSystem()
for n in range(21):
    print(f"{n} → {hdgl.hdgl_number(n)}")
